import { useState, useEffect } from 'react'
import { Award, Users, Award as Certified, Calendar, FileText, Star, Brain, History, Users as Clients, Video, User, Users as Group, Monitor, Phone, Mail, MapPin, Clock, TrendingUp } from 'lucide-react'
import ImageSlider from '../components/ImageSlider'
import './SessionsHistory.css'

const SessionsHistory = () => {
  const [sessions, setSessions] = useState({
    past: []
  })

  useEffect(() => {
    const savedSessions = localStorage.getItem('psychologistSessions')
    if (savedSessions) {
      const parsed = JSON.parse(savedSessions)
      setSessions(parsed)
    } else {
      const defaultSessions = {
        past: [
          {
            id: 1,
            date: '2025-03-15',
            type: 'Лично',
            topic: 'Стресс',
            rating: 5
          },
          {
            id: 2,
            date: '2025-03-10',
            type: 'Видео',
            topic: 'Отношения',
            rating: 4
          },
          {
            id: 3,
            date: '2025-03-05',
            type: 'Лично',
            topic: 'Карьера',
            rating: 5
          }
        ]
      }
      setSessions(defaultSessions)
      localStorage.setItem('psychologistSessions', JSON.stringify(defaultSessions))
    }

    const handleStorageChange = () => {
      const updated = localStorage.getItem('psychologistSessions')
      if (updated) {
        setSessions(JSON.parse(updated))
      }
    }

    window.addEventListener('storage', handleStorageChange)
    const interval = setInterval(handleStorageChange, 500)

    return () => {
      window.removeEventListener('storage', handleStorageChange)
      clearInterval(interval)
    }
  }, [])

  const sliderImages = [
    '/images/psychology-1.jpg',
    '/images/psychology-2.jpg',
    '/images/psychology-3.jpg',
    '/images/psychology-4.jpg'
  ]

  const handleRating = (sessionId, rating) => {
    const updatedSessions = {
      ...sessions,
      past: sessions.past.map(session => 
        session.id === sessionId ? { ...session, rating } : session
      )
    }
    setSessions(updatedSessions)
    localStorage.setItem('psychologistSessions', JSON.stringify(updatedSessions))
  }

  const stats = [
    { icon: Calendar, value: sessions.past.length, label: 'Всего сеансов', color: '#f59e0b' },
    { icon: Star, value: sessions.past.length > 0 ? (sessions.past.reduce((sum, s) => sum + (s.rating || 0), 0) / sessions.past.length).toFixed(1) : '0', label: 'Средний рейтинг', color: '#ec4899' },
    { icon: TrendingUp, value: '85%', label: 'Прогресс', color: '#10b981' }
  ]

  return (
    <div className="sessions-history">
      <div className="intro-panel">
        <div className="intro-panel-left">
          <div className="intro-label">Ваши консультации</div>
          <h1 className="intro-main-title">История сеансов</h1>
          <p className="intro-description">Просматривайте все ваши встречи с психологом, отслеживайте прогресс и оценивайте качество работы</p>
        </div>
        <div className="intro-panel-right">
          <div className="intro-metrics">
            <div className="intro-metric">
              <div className="metric-value">{sessions.past.length}</div>
              <div className="metric-label">Всего сеансов</div>
            </div>
            <div className="intro-metric">
              <div className="metric-value">{sessions.past.length > 0 ? (sessions.past.reduce((sum, s) => sum + (s.rating || 0), 0) / sessions.past.length).toFixed(1) : '0'}</div>
              <div className="metric-label">Средняя оценка</div>
            </div>
          </div>
        </div>
      </div>

      <ImageSlider images={sliderImages} />

      <div className="stats-section">
        {stats.map((stat, index) => {
          const IconComponent = stat.icon
          return (
            <div key={index} className="stat-card" style={{ '--stat-color': stat.color }}>
              <div className="stat-icon">
                <IconComponent className="stat-icon-img" size={32} />
              </div>
              <div className="stat-content">
                <div className="stat-value">{stat.value}</div>
                <div className="stat-label">{stat.label}</div>
              </div>
            </div>
          )
        })}
      </div>

      <div className="info-section">
        <div className="info-card">
          <div className="info-icon">
            <Award className="info-icon-img" size={32} />
          </div>
          <h3>15+ лет опыта</h3>
          <p>Профессиональная психологическая помощь</p>
        </div>
        <div className="info-card">
          <div className="info-icon">
            <Clients className="info-icon-img" size={32} />
          </div>
          <h3>5000+ клиентов</h3>
          <p>Довольных результатами терапии</p>
        </div>
        <div className="info-card">
          <div className="info-icon">
            <Certified className="info-icon-img" size={32} />
          </div>
          <h3>Сертифицированные специалисты</h3>
          <p>Высококвалифицированные психологи</p>
        </div>
      </div>

      <section className="sessions-section">
        <h2 className="section-title">
          <History className="section-icon" size={24} />
          История сеансов
        </h2>
        <div className="sessions-grid">
          {sessions.past.length > 0 ? (
            sessions.past.map((session) => (
              <div key={session.id} className="session-card">
                <div className="session-header">
                  <div className="session-type-badge" data-type={session.type.toLowerCase()}>
                    {session.type}
                  </div>
                  <div className="session-topic-badge">{session.topic}</div>
                </div>
                <div className="session-body">
                  <div className="session-detail">
                    <Calendar className="detail-icon" size={20} />
                    <span>{new Date(session.date).toLocaleDateString('ru-RU', { day: 'numeric', month: 'long', year: 'numeric' })}</span>
                  </div>
                  <div className="session-detail">
                    <FileText className="detail-icon" size={20} />
                    <span>{session.type === 'Лично' ? 'Очная консультация' : 'Онлайн-консультация'}</span>
                  </div>
                </div>
                <div className="rating-section">
                  <p className="rating-label">Оценка сеанса</p>
                  <div className="stars-container">
                    {[1, 2, 3, 4, 5].map((star) => (
                      <button
                        key={star}
                        className={`star-btn ${session.rating >= star ? 'filled' : ''}`}
                        onClick={() => handleRating(session.id, star)}
                      >
                        <Star className="star-icon" size={24} />
                      </button>
                    ))}
                  </div>
                </div>
              </div>
            ))
          ) : (
            <div className="empty-state">
              <Brain className="empty-icon" size={64} />
              <p>У вас пока нет завершенных сеансов</p>
            </div>
          )}
        </div>
      </section>

      <section className="services-section">
        <h2 className="section-title">
          <Users className="section-icon" size={24} />
          Наши услуги
        </h2>
        <div className="services-grid">
          <div className="service-card">
            <div className="service-icon">
              <User className="service-icon-img" size={32} />
            </div>
            <h3>Индивидуальная терапия</h3>
            <p>Персональный подход к решению ваших проблем</p>
          </div>
          <div className="service-card">
            <div className="service-icon">
              <Users className="service-icon-img" size={32} />
            </div>
            <h3>Семейная терапия</h3>
            <p>Работа с семейными отношениями и конфликтами</p>
          </div>
          <div className="service-card">
            <div className="service-icon">
              <Group className="service-icon-img" size={32} />
            </div>
            <h3>Групповые сеансы</h3>
            <p>Терапевтические группы для поддержки</p>
          </div>
          <div className="service-card">
            <div className="service-icon">
              <Monitor className="service-icon-img" size={32} />
            </div>
            <h3>Онлайн-консультации</h3>
            <p>Удобные видеосеансы из любой точки мира</p>
          </div>
        </div>
      </section>

      <section className="contact-section">
        <h2 className="section-title">Контакты</h2>
        <div className="contact-grid">
          <div className="contact-card">
            <Phone className="contact-card-icon" size={24} />
            <h3>Телефон</h3>
            <p>+7 (495) 123-45-67</p>
            <p>+7 (495) 123-45-68</p>
          </div>
          <div className="contact-card">
            <Mail className="contact-card-icon" size={24} />
            <h3>Email</h3>
            <p>info@psychoconsult.ru</p>
            <p>support@psychoconsult.ru</p>
          </div>
          <div className="contact-card">
            <MapPin className="contact-card-icon" size={24} />
            <h3>Адрес</h3>
            <p>г. Москва, ул. Психологическая, 10</p>
            <p>Ежедневно с 9:00 до 21:00</p>
          </div>
          <div className="contact-card">
            <Clock className="contact-card-icon" size={24} />
            <h3>Режим работы</h3>
            <p>Пн-Вс: 9:00 - 21:00</p>
            <p>Круглосуточная экстренная помощь</p>
          </div>
        </div>
      </section>
    </div>
  )
}

export default SessionsHistory

